import {
  Button,
  Col,
  Form,
  Input,
  InputNumber,
  Modal,
  Row,
  Select,
  Space,
  Switch,
} from 'antd';
import React, { useState, useEffect } from 'react';
import { useTranslation } from 'react-i18next';
import deliveryService from '../../services/delivery';
import Loading from '../../components/loading';
import { fetchDelivery } from '../../redux/slices/deliveries';
import { shallowEqual, useDispatch, useSelector } from 'react-redux';
import MediaUpload from '../../components/upload';
import Map from '../../components/map';
import getDefaultLocation from '../../helpers/getDefaultLocation';

const type_of_technique = [
  { label: 'Benzine', value: 'benzine' },
  { label: 'Diesel', value: 'diesel' },
  { label: 'Gas', value: 'gas' },
  { label: 'Motorbike', value: 'motorbike' },
  { label: 'Bike', value: 'bike' },
  { label: 'Foot', value: 'foot' },
  { label: 'Electric', value: 'electric' },
];

const DelivertSettingCreate = ({ data, handleCancel }) => {
  const { t } = useTranslation();
  const [form] = Form.useForm();
  const dispatch = useDispatch();
  const [loadingBtn, setLoadingBtn] = useState(false);
  const [loading, setLoading] = useState(false);
  const [image, setImage] = useState([]);
  const { settings } = useSelector(
    (state) => state.globalSettings,
    shallowEqual
  );
  const [location, setLocation] = useState(getDefaultLocation(settings));

  const createImages = (items) =>
    items.map((item) => ({
      uid: item.id,
      name: item.path,
      url: item.path,
    }));

  const fetchDeliverySettings = (id) => {
    setLoading(true);
    deliveryService
      .getById(id)
      .then((res) => {
        const responseData = {
          ...res.data,
          user_id: {
            label:
              res.data.deliveryMan.firstname +
              ' ' +
              res.data.deliveryMan.firstname,
            value: res.data.deliveryMan.id,
            images: createImages(res.data.galleries),
            location: {
              lat: res.data?.location?.latitude,
              lng: res.data?.location?.longitude,
            },
          },
        };
        setLocation({
          lat: res.data?.location?.latitude,
          lng: res.data?.location?.longitude,
        });
        setImage(createImages(res.data.galleries));
        form.setFieldsValue(responseData);
      })
      .finally(() => setLoading(false));
  };

  const onFinish = (values) => {
    setLoadingBtn(true);
    const params = {
      ...values,
      user_id: data.id,
      images: image.map((img) => img.name),
      location: {
        latitude: location.lat,
        longitude: location.lng,
      },
    };
    if (data.settingsId) {
      deliveryService
        .update(data.settingsId, params)
        .then(() => {
          handleCancel();
          dispatch(fetchDelivery());
        })
        .finally(() => setLoadingBtn(false));
    } else {
      deliveryService
        .create(params)
        .then(() => {
          handleCancel();
          dispatch(fetchDelivery());
        })
        .finally(() => setLoadingBtn(false));
    }
  };

  useEffect(() => {
    if (data.settingsId) {
      fetchDeliverySettings(data.settingsId);
    }
  }, []);

  return (
    <>
      <Modal
        visible={!!data}
        title={data.id ? t('edit.delivery.setting') : t('add.delivery.setting')}
        closable={false}
        style={{ minWidth: '80vw' }}
        footer={[
          <Space>
            <Button
              type='primary'
              htmlType='submit'
              key={'submit'}
              onClick={() => form.submit()}
              loading={loadingBtn}
            >
              {t('submit')}
            </Button>
            <Button type='default' key={'cancelBtn'} onClick={handleCancel}>
              {t('cancel')}
            </Button>
          </Space>,
        ]}
      >
        <Form
          name='basic'
          layout='vertical'
          onFinish={onFinish}
          form={form}
          initialValues={{ online: true }}
        >
          {loading ? (
            <Loading />
          ) : (
            <Row gutter={12}>
              <Col span={12}>
                <Form.Item
                  label={t('brand')}
                  name='brand'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value) {
                          return Promise.reject(new Error(t('required')));
                        } else if (value && value?.trim() === '') {
                          return Promise.reject(new Error(t('no.empty.space')));
                        } else if (value && value?.trim().length < 2) {
                          return Promise.reject(
                            new Error(t('must.be.at.least.2'))
                          );
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('model')}
                  name='model'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value) {
                          return Promise.reject(new Error(t('required')));
                        } else if (value && value?.trim() === '') {
                          return Promise.reject(new Error(t('no.empty.space')));
                        } else if (value && value?.trim().length < 2) {
                          return Promise.reject(
                            new Error(t('must.be.at.least.2'))
                          );
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('type.of.technique')}
                  name='type_of_technique'
                  rules={[
                    {
                      required: true,
                      message: t('required'),
                    },
                  ]}
                >
                  <Select options={type_of_technique} />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('car.number')}
                  name='number'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value) {
                          return Promise.reject(new Error(t('required')));
                        } else if (value && value?.trim() === '') {
                          return Promise.reject(new Error(t('no.empty.space')));
                        } else if (value && value?.trim().length < 2) {
                          return Promise.reject(
                            new Error(t('must.be.at.least.2'))
                          );
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('car.color')}
                  name='color'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value) {
                          return Promise.reject(new Error(t('required')));
                        } else if (value && value?.trim() === '') {
                          return Promise.reject(new Error(t('no.empty.space')));
                        } else if (value && value?.trim().length < 2) {
                          return Promise.reject(
                            new Error(t('must.be.at.least.2'))
                          );
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <Input />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('height')}
                  name='height'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value && value !== 0) {
                          return Promise.reject(new Error(t('required')));
                        } else if (!value || value < 1) {
                          return Promise.reject(new Error(t('min.1')));
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <InputNumber className='w-100' addonAfter='sm' />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('weight')}
                  name='kg'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value && value !== 0) {
                          return Promise.reject(new Error(t('required')));
                        } else if (!value || value < 1) {
                          return Promise.reject(new Error(t('min.1')));
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <InputNumber className='w-100' addonAfter='kg' />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('length')}
                  name='length'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value && value !== 0) {
                          return Promise.reject(new Error(t('required')));
                        } else if (!value || value < 1) {
                          return Promise.reject(new Error(t('min.1')));
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <InputNumber className='w-100' addonAfter='sm' />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('width')}
                  name='width'
                  rules={[
                    {
                      validator(_, value) {
                        if (!value && value !== 0) {
                          return Promise.reject(new Error(t('required')));
                        } else if (!value || value < 1) {
                          return Promise.reject(new Error(t('min.1')));
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <InputNumber className='w-100' addonAfter='sm' />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('image')}
                  name='images'
                  rules={[
                    {
                      validator() {
                        if (image?.length === 0) {
                          return Promise.reject(new Error(t('required')));
                        }
                        return Promise.resolve();
                      },
                    },
                  ]}
                >
                  <MediaUpload
                    type='deliveryman/settings'
                    imageList={image}
                    setImageList={setImage}
                    form={form}
                    length='1'
                    multiple={true}
                  />
                </Form.Item>
              </Col>
              <Col span={12}>
                <Form.Item
                  label={t('online')}
                  name='online'
                  valuePropName='checked'
                >
                  <Switch />
                </Form.Item>
              </Col>
              <Col span={24}>
                <Form.Item label={t('map')} name='location'>
                  <Map location={location} setLocation={setLocation} />
                </Form.Item>
              </Col>
            </Row>
          )}
        </Form>
      </Modal>
    </>
  );
};

export default DelivertSettingCreate;
